/************************************************************************************
* Header file for interface of the PHY layer.
* The functionality declared in this file all resides in the PHY layer of ZigBee
*
* (c) Copyright 2009, Freescale, Inc.  All rights reserved.
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
************************************************************************************/

#ifndef _PHY_H_
#define _PHY_H_

/************************************************************************************
*************************************************************************************
* Includes
*************************************************************************************
************************************************************************************/
#include "EmbeddedTypes.h"
#include "PortConfig.h"
#include "PhyPib.h"
#include "PhyTime.h"
#include "PhyISR.h"



/************************************************************************************
*************************************************************************************
* Public Definitions
*************************************************************************************
************************************************************************************/

#define gPhyMaxDataLength_c     255        
#define gPhyMinDataLength_c     2
#define gFillFifoBlockLength_c  8
#define gFifoThreshold_c        (gFillFifoBlockLength_c - 1)

//Radio: PHY states
enum {
  gIdle_c,
  gRX_c,
  gTX_c,
  gCCA_c,
  gED_c
};

//Radio: channel state
enum {
  gChannelIdle_c,
  gChannelBusy_c
};

//Radio: PHY requests exit states
enum {
  gPhySuccess_c,
  gPhyBusy_c,
  gPhyInvalidParam_c
};

//Radio: Cca types
enum {
  gCcaED_c,            
  gCcaCCA_MODE1_c,     
  gInvalidCcaType_c    
};

//Radio: PHY request action types
enum {
  gImmediateAction_c = 1<<0,
  gTimeTriggeredAction_c = 1<<1,
  gRxAckAction = 1<<2
};

/************************************************************************************
*************************************************************************************
* Public macros
*************************************************************************************
************************************************************************************/

#define PhyPlmeEdRequest(mode)           PhyPlmeCcaEdRequest(gCcaED_c, mode)
#define PhyPlmeCcaRequest(mode)          PhyPlmeCcaEdRequest(gCcaCCA_MODE1_c, mode)

/************************************************************************************
*************************************************************************************
* Public type definitions
*************************************************************************************
************************************************************************************/

typedef struct phyPacket_tag {
  uint8_t frameLength;
  uint8_t data[gPhyMaxDataLength_c];
} phyPacket_t;

typedef struct phyRxParams_tag {
  uint32_t    timeStamp;
  uint8_t     linkQuality;
} phyRxParams_t;

/************************************************************************************
*************************************************************************************
* Private type definitions
*************************************************************************************
************************************************************************************/


typedef struct phyTxIsrParameters_tag
{
  volatile uint8_t flags;
  uint8_t remainingLength;
  uint8_t macDataIndex;
  uint8_t fillFifoBlockLength;
  phyPacket_t *pTxPacket; 
} phyTxIsrParameters_t;


typedef struct phyRxIsrParameters_tag
{
  volatile uint8_t flags;
  uint8_t frameLength;
  uint8_t remainingLength;
  uint8_t macDataIndex;
  uint8_t unfillFifoBlockLength;
  phyPacket_t *pRxPacket;
  phyRxParams_t *pRxParams;
} phyRxIsrParameters_t;


typedef struct phyCcaIsrParameters_tag
{
  volatile uint8_t flags;
  uint16_t rssiAcc;
  uint16_t ccaMeasurementsCnt;
  uint8_t ccaThreshold;
  uint8_t ccaRssi;
  uint8_t ccaParam;
} phyCcaIsrParameters_t;


/************************************************************************************
*************************************************************************************
* Public prototypes
*************************************************************************************
************************************************************************************/

//MAC to PHY Interface functions
//PD/PLME
uint8_t PhyDataRequest(phyPacket_t *pTxPacket, uint8_t phyTxMode);
uint8_t PhyPlmeCcaEdRequest(uint8_t ccaType, uint8_t phyCcaMode);
uint8_t PhyPlmeRxRequest(phyPacket_t *pRxData, phyRxParams_t *pRxParams, uint8_t phyRxMode);
void    PhyPlmeForceTrxOffRequest(void);
void    PhyPlmeWaitRequest(void); // Forces Wait

// PHY to MAC Interface functions
// these functions should be at least defined as stubs in the upper layer

//PD/PLME
void    PhyPdDataConfirm(void);
void    PhyPdDataIndication(void);
void    PhyPlmeCcaConfirm(bool_t channelInUse);
void    PhyPlmeEdConfirm(uint8_t energyLevel);
void    PhyPlmeRxSFDDetectIndication(void);
void    PhyPlmeRxPHRErrorIndication(void);
void    PhyPlmeRxCRCErrorIndication(void);
void    PhyPlmeUnlockTx(void);
void    PhyPlmeUnlockRx(void);
void PhyPlmeB2BIndication(void);
//PHY-TIMING
void    PhyTimeTimeoutIndication(void);

//PHY initialization
void    InitializePhy(void);

/************************************************************************************
*************************************************************************************
* Private prototypes
*************************************************************************************
************************************************************************************/

//
uint8_t PhyPassTxParams(phyPacket_t *pTxPacket);
void PhyPassRxParams(phyPacket_t *pRxData, phyRxParams_t *pRxParams, uint8_t enableAckFilter);
void PhyPassCcaParams(uint8_t ccaParam);
uint8_t PhyGetState(void);
void PhySetState(uint8_t phyState);

#endif /* _PHY_H_ */
